/**
  *******************************************************************************
  * 
  * @file    hal_uart.c
  * @brief   uart module driver.
  *
  * @version v1.0
  * @date 22 Aug 2018
  * @author  AE Team
  * @note
  *
  * copyright (C) shanghai Eastsoft Mictroelectornics Co. Ltd. All rights reseverd.
  *
  @verbatim
  =================================================================================
  			##### How to use this driver #####
  =================================================================================
  [..]  The uart driver can be used as follows:
    (+) Initialize the Uart registers by using uart_init().
    (+) Transmit an amount of data using uart_send()
    (+) Receive an amount of data using uart_recv()

  @endverbatim
  *********************************************************************************
  */

#include <hic.h>
#include "hal_base.h"
#include "hal_uart.h"

/**
  * @addtogroup ES7P203_HAL
  * @{
  */

/**
  * @defgroup UART uart
  * @brief uart module driver
  * @{
  */

/**
  * @defgroup Uart_Public_Functions uart Public Functions
  * @brief    Initizlization and Data Transfers Functions
@verbatim
=================================================================================
		##### Uart Public Functions #####
=================================================================================
  [..]  The Uart driver can be used as follows:
    (+) Initialize the Uart registers by using uart_init().
    (+) Transmit an amount of data using uart_send()
    (+) Receive an amount of data using uart_recv()
@endverbatim
  * @{
  */

/**
  * @brief  Initialze the uart accroding to the specified baudrate.
  * @param  baud: specified baudrate.
  * @retval Status, see @ref hal_status_t.
  */
hal_status_t uart_init(uint32_t baud)
{
	uint16_t h_mode = 0;
	uint16_t l_mode = 0;

	h_mode = (get_system_clock() >> 4) / baud;
	l_mode = (get_system_clock() >> 6) / baud;

	if (h_mode <= 0) {
		return ERROR;
	}
	else if (h_mode <= 256) {
		BRGH0 = 1;
		BR0R  = h_mode;
		
	}
	else if (h_mode > 256) {
		if (l_mode <= 0) {
			return ERROR;
		}
		else if (l_mode <= 256) {
			BRGH0 = 0;
			BR0R  =l_mode;
		}
		else {
			return ERROR;
		}
	}
	else {}

	RX0EN  = 1;
	RX0LEN = 1;
	TX0EN  = 1;
	TX0LEN = 1;

	return OK;	
}

/**
  * @brief  Uart tx handler function.
  * @param  data: pointer of transmit data.
  * @retval None.
  */
void uart_tx(uint8_t *data)
{
	while (!TRMT0);
	TX0EN = 1;
	TX0B  = *data;
	while (!TRMT0);
	TX0EN = 0;

	return;
}

/**
  * @brief  Uart rx handler function.
  * @retval value: receive data.
  */
uint8_t uart_rx(void)
{
	return RX0B;
}

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
